import React from 'react';
import Navbar from '@/components/Navbar';
import Footer from '@/components/Footer';
import dbConnect from '@/lib/db';
import Event from '@/models/Event';
import { notFound } from 'next/navigation';
import Link from 'next/link';
import { Calendar, Clock, MapPin, Tag } from 'lucide-react';

export const revalidate = 60;

async function getEvent(slug: string) {
  await dbConnect();
  const event = await (Event as any).findOne({ slug, status: 'published' });
  if (!event) return null;
  return JSON.parse(JSON.stringify(event));
}

export default async function SingleEventPage(props: { params: Promise<{ slug: string }> }) {
  const params = await props.params;
  const event = await getEvent(params.slug);

  if (!event) return notFound();

  return (
    <div className="min-h-screen flex flex-col">
      <Navbar />
      
      <main className="flex-grow bg-white dark:bg-slate-900">
        {/* Header */}
        <div className="bg-slate-900 text-white py-20 relative overflow-hidden">
           <div className="absolute inset-0 opacity-20 bg-cover bg-center" style={{ backgroundImage: `url(${event.posterImage || event.image})` }}></div>
           <div className="absolute inset-0 bg-gradient-to-t from-slate-900 via-slate-900/80 to-slate-900/80"></div>
           
           <div className="max-w-4xl mx-auto px-4 relative z-10 text-center">
             <div className="inline-block bg-primary text-white text-xs font-bold px-3 py-1 rounded-full mb-6">
                {new Date(event.startDate) > new Date() ? 'Upcoming Event' : 'Past Event'}
             </div>
             <h1 className="text-3xl md:text-5xl font-bold font-display leading-tight mb-8">{event.title}</h1>
             
             <div className="flex flex-wrap justify-center gap-6 text-sm text-slate-300">
                <div className="flex items-center gap-2 bg-slate-800/50 px-4 py-2 rounded-full border border-slate-700">
                    <Calendar size={16} className="text-primary" /> 
                    {new Date(event.startDate).toLocaleDateString('en-GB', { weekday: 'short', day: 'numeric', month: 'long', year: 'numeric' })}
                </div>
                {event.timeText && (
                  <div className="flex items-center gap-2 bg-slate-800/50 px-4 py-2 rounded-full border border-slate-700">
                      <Clock size={16} className="text-primary" /> {event.timeText}
                  </div>
                )}
                {event.locationText && (
                  <div className="flex items-center gap-2 bg-slate-800/50 px-4 py-2 rounded-full border border-slate-700">
                      <MapPin size={16} className="text-primary" /> {event.locationText}
                  </div>
                )}
             </div>
           </div>
        </div>

        {/* Content */}
        <div className="max-w-4xl mx-auto px-4 py-16">
          <div className="grid md:grid-cols-3 gap-10">
             {/* Left: Content */}
             <div className="md:col-span-2">
                <h3 className="text-xl font-bold text-slate-900 dark:text-white mb-6 pb-4 border-b border-slate-100 dark:border-slate-800">Event Details</h3>
                <div 
                  className="prose prose-lg dark:prose-invert max-w-none"
                  dangerouslySetInnerHTML={{ __html: event.detailsContent || '<p>No details available.</p>' }}
                />
             </div>

             {/* Right: Sidebar Info */}
             <div>
                <div className="bg-slate-50 dark:bg-slate-800 p-6 rounded-xl border border-slate-200 dark:border-slate-700 sticky top-24">
                   {event.posterImage && (
                      <div className="mb-6 rounded-lg overflow-hidden shadow-md">
                         <img src={event.posterImage} alt="Poster" className="w-full h-auto" />
                      </div>
                   )}
                   <Link
                      href="/membership"
                      className="block w-full bg-primary hover:bg-primary-dark text-white text-center py-3 rounded-lg font-bold transition mb-4"
                    >
                      Join UKBCCI
                    </Link>
                </div>
             </div>
          </div>

          {/* Gallery */}
          {event.galleryImages && event.galleryImages.length > 0 && (
            <div className="mt-16 pt-16 border-t border-slate-200 dark:border-slate-800">
              <h3 className="text-2xl font-bold mb-8 text-slate-900 dark:text-white">Event Gallery</h3>
              <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
                {event.galleryImages.map((item: any, idx: number) => (
                  <div key={idx} className="rounded-lg overflow-hidden shadow-sm hover:shadow-md transition aspect-square">
                    <img src={item.url} alt={`Gallery ${idx + 1}`} loading="lazy" className="w-full h-full object-cover hover:scale-105 transition duration-500" />
                  </div>
                ))}
              </div>
            </div>
          )}
        </div>
      </main>

      <Footer />
    </div>
  );
}
